/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2008 Andy Jefferson - added bind check on whether class literal. Javadocs
2008 Andy Jefferson - cater for expression of field of an expression.
    ...
**********************************************************************/
package org.datanucleus.query.expression;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.Iterator;
import java.util.List;

import org.datanucleus.exceptions.ClassNotResolvedException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.query.symbol.PropertySymbol;
import org.datanucleus.query.symbol.Symbol;
import org.datanucleus.query.symbol.SymbolTable;

/**
 * Expression for a primary object. 
 * This may be a field, or an explicit variable/parameter, or a field invoked on an expression.
 */
public class PrimaryExpression extends Expression
{
    SymbolTable symtbl;

    /** The components of the expression. e.g "a.b.c" will have "a", "b", "c". */
    List<String> tuples;

    /**
     * PrimaryExpression made up of a series of field names.
     * e.g "a.b.c"
     * @param symtbl Symbol Table
     * @param tuples The components of the expression
     */
    public PrimaryExpression(SymbolTable symtbl, List tuples)
    {
        this.symtbl = symtbl;
        this.tuples = tuples;
    }

    /**
     * PrimaryExpression on an expression.
     * e.g "((B)a).c" so the left expression is a CastExpression, and the tuples are "c".
     * @param symtbl Symbol Table
     * @param left The left expression
     * @param tuples The tuples of the primary
     */
    public PrimaryExpression(SymbolTable symtbl, Expression left, List tuples)
    {
        this.symtbl = symtbl;
        this.left = left;
        this.tuples = tuples;
    }

    /**
     * Accessor for the expression "id". This will be something like "a.b.c".
     * @return The id
     */
    public String getId()
    {
        StringBuffer id = new StringBuffer();
        for (int i=0; i<tuples.size(); i++)
        {
            if (id.length() > 0)
            {
                id.append('.');
            }
            id.append(tuples.get(i));
        }
        return id.toString();
    }

    public List<String> getTuples()
    {
        return tuples;
    }

    /**
     * Method to bind the expression to the symbol table as appropriate.
     * @return The symbol for this expression
     */
    public Symbol bind()
    {
        if (left != null)
        {
            left.bind();
        }

        if (symtbl.hasSymbol(getId()))
        {
            symbol = symtbl.getSymbol(getId());
        }
        else
        {
            if (left != null)
            {
                // TODO Cater for finding field of left expression
                return null;
            }

            try
            {
                Class symbolType = symtbl.getType(tuples);
                symbol = new PropertySymbol(getId(), symbolType);
            }
            catch (NucleusUserException nue)
            {
                // Thrown if a field in the primary expression doesn't exist.
                // This may be due to an entry like "org.jpox.samples.MyClass" used for "instanceof"
                StringBuffer str = new StringBuffer();
                Iterator<String> iter = tuples.iterator();
                while (iter.hasNext())
                {
                    String tuple = iter.next();
                    if (str.length() > 0)
                    {
                        str.append('.');
                    }
                    str.append(tuple);
                }
                String className = str.toString();

                try
                {
                    // Try to find this as a complete class name (e.g as used in "instanceof")
                    Class cls = symtbl.getSymbolResolver().resolveClass(className);

                    // Represents a valid class so throw exception to get the PrimaryExpression swapped
                    throw new PrimaryExpressionIsClassLiteralException(cls);
                }
                catch (ClassNotResolvedException cnre)
                {
                    // Try to find classname.staticField
                    if (className.indexOf('.') < 0)
                    {
                        // {candidateCls}.primary so "primary" is staticField ?
                        Class primaryCls = symtbl.getSymbolResolver().getPrimaryClass();
                        if (primaryCls == null)
                        {
                            throw new NucleusUserException("Class name " + className + " could not be resolved");
                        }
                        try
                        {
                            Field fld = primaryCls.getDeclaredField(className);
                            if (!Modifier.isStatic(fld.getModifiers()))
                            {
                                throw new NucleusUserException("Identifier " + className + " is unresolved (not a static field)");
                            }
                            throw new PrimaryExpressionIsClassStaticFieldException(fld);
                        }
                        catch (NoSuchFieldException nsfe)
                        {
                            if (symtbl.getSymbolResolver().supportsImplicitVariables() && left == null)
                            {
                                // Implicit variable assumed so swap this primary for it
                                throw new PrimaryExpressionIsVariableException(symtbl, className);
                            }
                            throw new NucleusUserException("Class name " + className + " could not be resolved");
                        }
                    }
                    else
                    {
                        try
                        {
                            String staticFieldName = className.substring(className.lastIndexOf('.')+1);
                            className = className.substring(0, className.lastIndexOf('.'));
                            Class cls = symtbl.getSymbolResolver().resolveClass(className);
                            try
                            {
                                Field fld = cls.getDeclaredField(staticFieldName);
                                if (!Modifier.isStatic(fld.getModifiers()))
                                {
                                    throw new NucleusUserException("Identifier " + className + "." + staticFieldName + " is unresolved (not a static field)");
                                }
                                throw new PrimaryExpressionIsClassStaticFieldException(fld);
                            }
                            catch (NoSuchFieldException nsfe)
                            {
                                throw new NucleusUserException("Identifier " + className + "." + staticFieldName + " is unresolved (not a static field)");
                            }
                        }
                        catch (ClassNotResolvedException cnre2)
                        {
                            if (symtbl.getSymbolResolver().supportsImplicitVariables() && left == null)
                            {
                                // Implicit variable, so put as "left" and remove from tuples
                                VariableExpression varExpr = new VariableExpression(symtbl, className);
                                varExpr.bind();
                                left = varExpr;
                                tuples.remove(0);
                            }
                            else
                            {
                                // Just throw the original exception
                                throw nue;
                            }
                        }
                    }
                }
            }
        }
        return symbol;
    }

    /**
     * Accessor for string form of the expression.
     * Returns something like "PrimaryExpression {a.b.c}" when left is null, or 
     * "PrimaryExpression {ParameterExpression {a}.b.c}" when left is the ParameterExpression
     */
    public String toString()
    {
        if (left != null)
        {
            return "PrimaryExpression{" + left + "." + getId() + "}";
        }
        else
        {
            return "PrimaryExpression{" + getId() + "}";
        }
    }
}