/**********************************************************************
Copyright (c) 2005 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;

import org.datanucleus.api.ApiAdapter;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.store.mapped.DatastoreClass;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * An expression representing a class.
 * This is used as follows :-
 * <UL>
 * <LI>JDOQL : access public static final fields. Here we invoke accessField on the ClassExpression.</LI>
 * <LI>JDOQL : use the "instanceof" operator. Here we invoke instanceOf on an ObjectExpression pass in a ClassExpression</LI>
 * <LI>JPQL : process the FROM candidates so each candidate is a ClassExpression. 
 *     Here we set the join(s) for later processing</LI>
 * </UL>
 */
public class ClassExpression extends ScalarExpression
{
    /** The class being represented. */
    private Class cls;

    /** Joins defined for this class. **/
    private List joinExprs;

    /**
     * Constructor.
     * @param qs The Query Statement
     * @param cls The class
     **/
    public ClassExpression(QueryExpression qs, Class cls)
    {
        super(qs);

        this.cls = cls;
    }

    /**
     * Accessor for the class being represented.
     * @return The class
     */
    public Class getCls()
    {
        return cls;
    }

    /**
     * Method called when wanting to call public static final methods on the class.
     * @param fieldName Name of the public static final field
     * @param innerJoin Not used
     * @return Expression for the field access
     */
    public ScalarExpression accessField(String fieldName, boolean innerJoin)
    {
        try
        {
            Field fld = cls.getField(fieldName);
            if (!Modifier.isStatic(fld.getModifiers()) || !Modifier.isFinal(fld.getModifiers()) || !Modifier.isPublic(fld.getModifiers()))
            {
                throw new NucleusUserException(LOCALISER.msg("037008", fieldName, cls.getName()));
            }
            Object value = fld.get(null);
            if (value == null)
            {
                return new NullLiteral(qs);
            }

            JavaTypeMapping m = null;
            ApiAdapter api = qs.getStoreManager().getApiAdapter();
            if (api.isPersistable(cls))
            {
                // PC class, so maybe has its own table
                try
                {
                    // PC class, so maybe has its own table
                    DatastoreClass clsTable = qs.getStoreManager().getDatastoreClass(cls.getName(), qs.getClassLoaderResolver());
                    m = clsTable.getMemberMapping(fieldName);
                }
                catch (Exception e)
                {
                    // PC class has no table or no such field so just get a default mapping for type
                    m = qs.getStoreManager().getMappingManager().getMappingWithDatastoreMapping(
                        value.getClass(), false, false, qs.getClassLoaderResolver());
                }
            }
            else
            {
                // Just get a default mapping for type
                m = qs.getStoreManager().getMappingManager().getMappingWithDatastoreMapping(
                    value.getClass(), false, false, qs.getClassLoaderResolver());
            }
            return m.newLiteral(qs, value);
        }
        catch (IllegalAccessException iae)
        {
            // Will not happen since we already checked for it
        }
        catch (NoSuchFieldException nsfe)
        {
            throw new NucleusUserException(LOCALISER.msg("037009", fieldName, cls.getName()));
        }
        return null;
    }

    /**
     * Method to add a join expression for this class.
     * @param expr Expression to join to
     * @return This expression
     */
    public ScalarExpression join(JoinExpression expr)
    {
        if (joinExprs == null)
        {
            joinExprs = new ArrayList(); // Joins order has to be preserved
        }
        joinExprs.add(expr);
        return this;
    }

    /**
     * Accessor for the join expression(s) for this class.
     * @return Join expressions
     */
    public JoinExpression[] getJoins()
    {
        if (joinExprs == null)
        {
            return null;
        }
        return (JoinExpression[])joinExprs.toArray(new JoinExpression[joinExprs.size()]);
    }
}