/**********************************************************************
Copyright (c) 2012 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
   ...
**********************************************************************/
package org.datanucleus.api.jpa;

import java.sql.Time;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.persistence.EntityManager;
import javax.persistence.FlushModeType;
import javax.persistence.LockModeType;
import javax.persistence.Parameter;
import javax.persistence.PersistenceException;
import javax.persistence.Query;
import javax.persistence.QueryTimeoutException;
import javax.persistence.TemporalType;

import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.metadata.StoredProcQueryParameterMode;
import org.datanucleus.store.query.AbstractStoredProcedureQuery;
import org.datanucleus.store.query.NoQueryResultsException;
import org.datanucleus.store.query.QueryInvalidParametersException;
import org.datanucleus.util.Localiser;

import javax.persistence.jpa21.ParameterMode;
import javax.persistence.jpa21.StoredProcedureQuery;

/**
 * Implementation of a StoredProcedureQuery.
 * Wraps an internal query.
 * TODO Extend JPAQuery
 */
public class JPAStoredProcedureQuery /*extends JPAQuery */implements StoredProcedureQuery
{
    /** Localisation utility for output messages */
    protected static final Localiser LOCALISER = Localiser.getInstance(
        "org.datanucleus.api.jpa.Localisation", NucleusJPAHelper.class.getClassLoader());

    /** Underlying EntityManager handling persistence. */
    EntityManager em;

    /** Procedure name. */
    String procName;

    /** Underlying query providing the querying capability. */
    org.datanucleus.store.query.AbstractStoredProcedureQuery query;

    /** Flush mode for the query. */
    FlushModeType flushMode = FlushModeType.AUTO;

    /** Lock mode for the query. */
    LockModeType lockMode = null;

    /** The current start position. */
    private long startPosition = 0;

    /** The current max number of results. */
    private long maxResults = Long.MAX_VALUE;

    public JPAStoredProcedureQuery(EntityManager em, org.datanucleus.store.query.Query query)
    {
        this.em = em;
        this.query = (AbstractStoredProcedureQuery) query;
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#getFlushMode()
     */
    public FlushModeType getFlushMode()
    {
        return flushMode;
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#setFlushMode(javax.persistence.FlushModeType)
     */
    public StoredProcedureQuery setFlushMode(FlushModeType mode)
    {
        flushMode = mode;
        return this;
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#getLockMode()
     */
    public LockModeType getLockMode()
    {
        return lockMode;
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#setLockMode(javax.persistence.LockModeType)
     */
    public Query setLockMode(LockModeType arg0)
    {
        throw new IllegalStateException("Stored Procedures don't support locking");
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#getHints()
     */
    public Map<String, Object> getHints()
    {
        Map extensions = query.getExtensions();
        Map map = new HashMap();
        if (extensions != null && extensions.size() > 0)
        {
            map.putAll(extensions);
        }
        return map;
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#setHint(java.lang.String, java.lang.Object)
     */
    public StoredProcedureQuery setHint(String hintName, Object value)
    {
        if (hintName != null && hintName.equalsIgnoreCase("javax.persistence.query.timeout"))
        {
            query.setDatastoreReadTimeoutMillis((Integer)value);
        }
        if (hintName != null && hintName.equalsIgnoreCase("datanucleus.query.fetchSize"))
        {
            if (value instanceof Integer)
            {
                query.getFetchPlan().setFetchSize((Integer)value);
            }
            else if (value instanceof Long)
            {
                query.getFetchPlan().setFetchSize(((Long)value).intValue());
            }
        }

        // Just treat a "hint" as an "extension".
        query.addExtension(hintName, value);
        return this;
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#getParameter(int, java.lang.Class)
     */
    public <T> Parameter<T> getParameter(int position, Class<T> type)
    {
        Set paramKeys = query.getImplicitParameters().keySet();
        Iterator iter = paramKeys.iterator();
        while (iter.hasNext())
        {
            Object paramKey = iter.next();
            if (paramKey instanceof Integer && ((Integer)paramKey).intValue() == position)
            {
                Object value = query.getImplicitParameters().get(paramKey);
                if (value != null && type.isAssignableFrom(value.getClass()))
                {
                    return new JPAQueryParameter((Integer)paramKey, type);
                }
            }
        }
        throw new IllegalArgumentException("No parameter at position=" + position + " and type=" + type.getName());
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#getParameter(int)
     */
    public Parameter<?> getParameter(int position)
    {
        Set paramKeys = query.getImplicitParameters().keySet();
        Iterator iter = paramKeys.iterator();
        while (iter.hasNext())
        {
            Object paramKey = iter.next();
            if (paramKey instanceof Integer && ((Integer)paramKey).intValue() == position)
            {
                Object value = query.getImplicitParameters().get(paramKey);
                return new JPAQueryParameter((Integer)paramKey, value != null ? value.getClass() : null);
            }
        }
        throw new IllegalArgumentException("No parameter at position=" + position);
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#getParameter(java.lang.String, java.lang.Class)
     */
    public <T> Parameter<T> getParameter(String name, Class<T> type)
    {
        Set paramKeys = query.getImplicitParameters().keySet();
        Iterator iter = paramKeys.iterator();
        while (iter.hasNext())
        {
            Object paramKey = iter.next();
            if (paramKey instanceof String && ((String)paramKey).equals(name))
            {
                Object value = query.getImplicitParameters().get(paramKey);
                if (value != null && type.isAssignableFrom(value.getClass()))
                {
                    return new JPAQueryParameter((String)paramKey, type);
                }
            }
        }
        throw new IllegalArgumentException("No parameter with name " + name + " and type=" + type.getName());
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#getParameter(java.lang.String)
     */
    public Parameter<?> getParameter(String name)
    {
        Set paramKeys = query.getImplicitParameters().keySet();
        Iterator iter = paramKeys.iterator();
        while (iter.hasNext())
        {
            Object paramKey = iter.next();
            if (paramKey instanceof String && ((String)paramKey).equals(name))
            {
                Object value = query.getImplicitParameters().get(paramKey);
                return new JPAQueryParameter((String)paramKey, value != null ? value.getClass() : null);
            }
        }
        throw new IllegalArgumentException("No parameter with name " + name);
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#getParameters()
     */
    public Set<Parameter<?>> getParameters()
    {
        Set paramKeys = query.getImplicitParameters().keySet();
        Set<Parameter<?>> parameters = new HashSet();
        Iterator iter = paramKeys.iterator();
        while (iter.hasNext())
        {
            Object paramKey = iter.next();
            Object value = query.getImplicitParameters().get(paramKey);
            if (paramKey instanceof String)
            {
                parameters.add(new JPAQueryParameter((String)paramKey, value != null ? value.getClass() : null));
            }
            else if (paramKey instanceof Integer)
            {
                parameters.add(new JPAQueryParameter((Integer)paramKey, value != null ? value.getClass() : null));
            }
        }
        return parameters;
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#getParameterValue(int)
     */
    public Object getParameterValue(int position)
    {
        if (query.getImplicitParameters().containsKey(position))
        {
            return query.getImplicitParameters().get(position);
        }
        return null;
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#getParameterValue(javax.persistence.Parameter)
     */
    public <T> T getParameterValue(Parameter<T> param)
    {
        if (param.getName() != null)
        {
            if (query.getImplicitParameters().containsKey(param.getName()))
            {
                return (T)query.getImplicitParameters().get(param.getName());
            }
        }
        else
        {
            if (query.getImplicitParameters().containsKey(param.getPosition()))
            {
                return (T)query.getImplicitParameters().get(param.getPosition());
            }
        }
        throw new IllegalStateException("No parameter matching " + param + " bound to this query");
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#getParameterValue(java.lang.String)
     */
    public Object getParameterValue(String name)
    {
        if (query.getImplicitParameters().containsKey(name))
        {
            return query.getImplicitParameters().get(name);
        }
        return null;
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#isBound(javax.persistence.Parameter)
     */
    public boolean isBound(Parameter<?> param)
    {
        if (param.getName() != null)
        {
            if (query.getImplicitParameters().containsKey(param.getName()))
            {
                return true;
            }
        }
        else
        {
            if (query.getImplicitParameters().containsKey(param.getPosition()))
            {
                return true;
            }
        }
        return false;
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#unwrap(java.lang.Class)
     */
    public <T> T unwrap(Class<T> cls)
    {
        if (cls == org.datanucleus.store.query.Query.class)
        {
            return (T)query;
        }
        throw new PersistenceException("Not supported unwrapping of query to " + cls.getName());
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#setParameter(javax.persistence.Parameter, java.lang.Object)
     */
    public <T> StoredProcedureQuery setParameter(Parameter<T> param, T value)
    {
        if (param.getName() != null)
        {
            setParameter(param.getName(), value);
        }
        else
        {
            setParameter(param.getPosition(), value);
        }
        return this;
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#setParameter(javax.persistence.Parameter, java.util.Calendar, javax.persistence.TemporalType)
     */
    public StoredProcedureQuery setParameter(Parameter<Calendar> param, Calendar cal, TemporalType type)
    {
        if (param.getName() != null)
        {
            setParameter(param.getName(), cal, type);
        }
        else
        {
            setParameter(param.getPosition(), cal, type);
        }
        return this;
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#setParameter(javax.persistence.Parameter, java.util.Date, javax.persistence.TemporalType)
     */
    public StoredProcedureQuery setParameter(Parameter<Date> param, Date date, TemporalType type)
    {
        if (param.getName() != null)
        {
            setParameter(param.getName(), date, type);
        }
        else
        {
            setParameter(param.getPosition(), date, type);
        }
        return this;
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#setParameter(java.lang.String, java.lang.Object)
     */
    public StoredProcedureQuery setParameter(String name, Object value)
    {
        try
        {
            query.setImplicitParameter(name, value);
        }
        catch (QueryInvalidParametersException ipe)
        {
            throw new IllegalArgumentException(ipe.getMessage(), ipe);
        }
        return this;
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#setParameter(java.lang.String, java.util.Calendar, javax.persistence.TemporalType)
     */
    public StoredProcedureQuery setParameter(String name, Calendar value, TemporalType temporalType)
    {
        Object paramValue = value;
        if (temporalType == TemporalType.DATE)
        {
            paramValue = value.getTime();
        }
        else if (temporalType == TemporalType.TIME)
        {
            paramValue = new Time(value.getTime().getTime());
        }
        else if (temporalType == TemporalType.TIMESTAMP)
        {
            paramValue = new Timestamp(value.getTime().getTime());
        }

        try
        {
            query.setImplicitParameter(name, paramValue);
        }
        catch (QueryInvalidParametersException ipe)
        {
            throw new IllegalArgumentException(ipe.getMessage());
        }
        return this;
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#setParameter(java.lang.String, java.util.Date, javax.persistence.TemporalType)
     */
    public StoredProcedureQuery setParameter(String name, Date value, TemporalType temporalType)
    {
        Object paramValue = value;
        if (temporalType == TemporalType.TIME && !(value instanceof Time))
        {
            paramValue = new Time(value.getTime());
        }
        else if (temporalType == TemporalType.TIMESTAMP && !(value instanceof Timestamp))
        {
            paramValue = new Timestamp(value.getTime());
        }

        try
        {
            query.setImplicitParameter(name, paramValue);
        }
        catch (QueryInvalidParametersException ipe)
        {
            throw new IllegalArgumentException(ipe.getMessage());
        }
        return this;
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#setParameter(int, java.lang.Object)
     */
    public StoredProcedureQuery setParameter(int position, Object value)
    {
        try
        {
            query.setImplicitParameter(position, value);
        }
        catch (QueryInvalidParametersException ipe)
        {
            throw new IllegalArgumentException(ipe.getMessage(), ipe);
        }
        return this;
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#setParameter(int, java.util.Calendar, javax.persistence.TemporalType)
     */
    public StoredProcedureQuery setParameter(int position, Calendar value, TemporalType temporalType)
    {
        Object paramValue = value;
        if (temporalType == TemporalType.DATE)
        {
            paramValue = value.getTime();
        }
        else if (temporalType == TemporalType.TIME)
        {
            paramValue = new Time(value.getTime().getTime());
        }
        else if (temporalType == TemporalType.TIMESTAMP)
        {
            paramValue = new Timestamp(value.getTime().getTime());
        }

        try
        {
            query.setImplicitParameter(position, paramValue);
        }
        catch (QueryInvalidParametersException ipe)
        {
            throw new IllegalArgumentException(ipe.getMessage());
        }
        return this;
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#setParameter(int, java.util.Date, javax.persistence.TemporalType)
     */
    public StoredProcedureQuery setParameter(int position, Date value, TemporalType temporalType)
    {
        Object paramValue = value;
        if (temporalType == TemporalType.TIME && !(value instanceof Time))
        {
            paramValue = new Time(value.getTime());
        }
        else if (temporalType == TemporalType.TIMESTAMP && !(value instanceof Timestamp))
        {
            paramValue = new Timestamp(value.getTime());
        }

        try
        {
            query.setImplicitParameter(position, paramValue);
        }
        catch (QueryInvalidParametersException ipe)
        {
            throw new IllegalArgumentException(ipe.getMessage());
        }
        return this;
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#registerStoredProcedureParameter(int, java.lang.Class, jpa_2_1.ParameterMode)
     */
    public StoredProcedureQuery registerStoredProcedureParameter(int position, Class type, ParameterMode mode)
    {
        StoredProcQueryParameterMode paramMode = null;
        if (mode == ParameterMode.IN)
        {
            paramMode = StoredProcQueryParameterMode.IN;
        }
        else if (mode == ParameterMode.OUT)
        {
            paramMode = StoredProcQueryParameterMode.OUT;
        }
        else if (mode == ParameterMode.INOUT)
        {
            paramMode = StoredProcQueryParameterMode.INOUT;
        }
        else if (mode == ParameterMode.REF_CURSOR)
        {
            paramMode = StoredProcQueryParameterMode.REF_CURSOR;
        }
        query.registerParameter(position, type, paramMode);
        return this;
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#registerStoredProcedureParameter(java.lang.String, java.lang.Class, jpa_2_1.ParameterMode)
     */
    public StoredProcedureQuery registerStoredProcedureParameter(String parameterName, Class type, ParameterMode mode)
    {
        StoredProcQueryParameterMode paramMode = null;
        if (mode == ParameterMode.IN)
        {
            paramMode = StoredProcQueryParameterMode.IN;
        }
        else if (mode == ParameterMode.OUT)
        {
            paramMode = StoredProcQueryParameterMode.OUT;
        }
        else if (mode == ParameterMode.INOUT)
        {
            paramMode = StoredProcQueryParameterMode.INOUT;
        }
        else if (mode == ParameterMode.REF_CURSOR)
        {
            paramMode = StoredProcQueryParameterMode.REF_CURSOR;
        }
        query.registerParameter(parameterName, type, paramMode);
        return this;
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#getOutputParameterValue(int)
     */
    public Object getOutputParameterValue(int position)
    {
        return query.getOutputParameterValue(position);
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#getOutputParameterValue(java.lang.String)
     */
    public Object getOutputParameterValue(String parameterName)
    {
        return query.getOutputParameterValue(parameterName);
    }

    boolean executeProcessed = false;

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#execute()
     */
    public boolean execute()
    {
        Object hasResultSet = query.execute();
        executeProcessed = true;
        return (Boolean)hasResultSet;
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#hasMoreResults()
     */
    public boolean hasMoreResults()
    {
        if (executeProcessed)
        {
            return query.hasMoreResults();
        }
        return false;
    }

    /* (non-Javadoc)
     * @see jpa_2_1.StoredProcedureQuery#getUpdateCount()
     */
    public int getUpdateCount()
    {
        if (executeProcessed)
        {
            return query.getUpdateCount();
        }
        return -1;
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#executeUpdate()
     */
    public int executeUpdate()
    {
        try
        {
            if (flushMode == FlushModeType.AUTO && em.getTransaction().isActive())
            {
                em.flush();
            }

            Boolean hasResultSet = (Boolean)query.execute();
            if (hasResultSet)
            {
                throw new IllegalStateException("Stored procedure returned a result set but method requires an update count");
            }
            return query.getUpdateCount();
        }
        catch (NoQueryResultsException nqre)
        {
            return 0;
        }
        catch (org.datanucleus.store.query.QueryTimeoutException qte)
        {
            throw new QueryTimeoutException();
        }
        catch (NucleusException jpe)
        {
            throw NucleusJPAHelper.getJPAExceptionForNucleusException(jpe);
        }
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#getResultList()
     */
    public List getResultList()
    {
        if (executeProcessed)
        {
            return (List)query.getNextResults();
        }

        try
        {
            if (flushMode == FlushModeType.AUTO && em.getTransaction().isActive())
            {
                em.flush();
            }

            Boolean hasResultSet = (Boolean)query.execute();
            if (!hasResultSet)
            {
                throw new IllegalStateException("Stored proc should have returned result set but didnt");
            }
            return (List)query.getNextResults();
        }
        catch (NoQueryResultsException nqre)
        {
            return null;
        }
        catch (org.datanucleus.store.query.QueryTimeoutException qte)
        {
            throw new QueryTimeoutException();
        }
        catch (NucleusException jpe)
        {
            throw NucleusJPAHelper.getJPAExceptionForNucleusException(jpe);
        }
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#getSingleResult()
     */
    public Object getSingleResult()
    {
        if (executeProcessed)
        {
            query.setUnique(true);
            return query.getNextResults();
        }

        try
        {
            if (flushMode == FlushModeType.AUTO && em.getTransaction().isActive())
            {
                em.flush();
            }

            query.setUnique(true);
            Boolean hasResultSet = (Boolean)query.execute();
            if (!hasResultSet)
            {
                throw new IllegalStateException("Stored proc should have returned result set but didnt");
            }
            return query.getNextResults();
        }
        catch (NoQueryResultsException nqre)
        {
            return null;
        }
        catch (org.datanucleus.store.query.QueryTimeoutException qte)
        {
            throw new QueryTimeoutException();
        }
        catch (NucleusException jpe)
        {
            throw NucleusJPAHelper.getJPAExceptionForNucleusException(jpe);
        }
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#getFirstResult()
     */
    public int getFirstResult()
    {
        return (int)query.getRangeFromIncl();
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#getMaxResults()
     */
    public int getMaxResults()
    {
        long queryMin = query.getRangeFromIncl();
        long queryMax = query.getRangeToExcl();
        long max = queryMax - queryMin;
        if (max > Integer.MAX_VALUE)
        {
            return Integer.MAX_VALUE;
        }
        return (int)max;
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#setFirstResult(int)
     */
    public Query setFirstResult(int startPosition)
    {
        if (startPosition < 0)
        {
            throw new IllegalArgumentException(LOCALISER.msg("Query.StartPositionInvalid"));
        }

        this.startPosition = startPosition;
        if (this.maxResults == Long.MAX_VALUE)
        {
            query.setRange(this.startPosition, Long.MAX_VALUE);
        }
        else
        {
            query.setRange(this.startPosition, this.startPosition+this.maxResults);
        }
        return this;
    }

    /* (non-Javadoc)
     * @see javax.persistence.Query#setMaxResults(int)
     */
    public Query setMaxResults(int max)
    {
        if (max < 0)
        {
            throw new IllegalArgumentException(LOCALISER.msg("Query.MaxResultsInvalid"));
        }

        this.maxResults = max;
        query.setRange(startPosition, startPosition+max);
        return this;
    }
}